/**
  * raudio 2020 - webradio for html5-audio
  * this is pentode v1
  * 2020 Andreas Bahr
  * GPL3 - see license.txt
  */

var rau = new Audio();
var rauStation = "";
var rauStream = "";
var rauStationData;
var rauStreamData;
var rauShowOnceMSG = "";

/** client data connector */
var rauData = {

	/** load station data via ajax */
	loadStationData: function() {
		$.ajax({
			url: "system/connector.php",
			type: "get",
			data: "cmd=LIST&clienttype=" + rauToolbox.clientName() + "-HTML5-" + rauToolbox.clientOS() + "&mimetype=" + rauDio.checkAudio(),
			datatype: 'json',
			success: function(data) {
				if(data.cmd == "OK") {
					rauStationData = data.station;
					$("#stationButtons").html(rauGui.drawStationButtons());
					rauGuiCtrl.resetGUI();
					rauGuiCtrl.setupStatus();
					rauGuiCtrl.setupMute();
					rauGuiCtrl.setupStations();
				} else {
					$("#displaySub").text("Error: Server malfunction");
				}
			},
			error:function() {
				$("#displaySub").text("Error: Server not found");
			}   
		}); 
	},

	/** load stream data via ajax */
	loadStreamData: function(id) {
		var stationName = ""; var stationLongName = "";
		
		$.ajax({
			url: "system/connector.php",
			type: "get",
			data: "cmd=TUNE&id=" + id + "clienttype=" + rauToolbox.clientName() + "-HTML5-" + rauToolbox.clientOS() + "&mimetype=" + rauDio.checkAudio(),
			datatype: 'json',
			success: function(data) {
				if(data.cmd == "OK") {
					rauStreamData = data.station;
					$("#streamButtons").html(rauGui.drawStreamButtons());
					rauGuiCtrl.setupStatus();
					rauGuiCtrl.setupMute();
					rauGuiCtrl.setupStreams();
					rauDio.tuneStation(id, rauStreamData);
				} else {
					$("#displaySub").text("Error: Server malfunction");
				}
			},
			error:function() {
				$("#displaySub").text("Error: Server not found");
			}   
		}); 
	},

	/** show system info */
	loadSystemInfo: function() {
		var serverinfos;
		$.ajax({
			url: "system/connector.php",
			type: "get",
			data: "cmd=INFO-PARAMS&clienttype=" + rauToolbox.clientName() + "-HTML5-" + rauToolbox.clientOS() + "&mimetype=" + rauDio.checkAudio(),
			datatype: 'json',
			success: function(data) {
				if(data.cmd == "OK") {
					serverinfos = data.info;
					alert(serverinfos.system + "\n" + serverinfos.signature + "\n" + "Client-Type: " + serverinfos.echoClientType + "\n" + "Codec Mime-Types: " + serverinfos.echoMimeType);
				} else {
					alert("Error: Server malfunction");
				}
			},
			error:function() {
				alert("Error: Server not found");
			}   
		}); 
	},

	/** get editor-link */
	loadEditorLink: function() {
		$.ajax({
			url: "system/connector.php",
			type: "get",
			data: "cmd=EDITOR-LINK",
			datatype: 'json',
			success: function(data) {
				if(data.cmd == "OK") {
					rauGui.showEditorLink(data.linkUrl);
				}
			},
			error:function() {
				//alert("Error: Server not found");
			}   
		}); 
	},

	/** query station data */
	queryStationData: function(stationId, attr) {
		var result = "";
		$.each(rauStationData, function(key, values) {
			if(values.id == stationId) {
				$.each(values, function(stationKey, stationValues) {
					if(stationKey == attr) {
						result = stationValues;
					}
				});
			}
		});
		return result;
	},
	
	/** station name formula */
	selectStationName: function(name, longName) {
		var fullName = "";
		if(longName != "") {
			fullName = longName;
		} else {
			fullName = name;
		}
		return fullName;
	},
};

/** audio stuff */
var rauDio = {

	/** check audio capability */
	checkAudio: function() {
		var audioCaps = "";
		const queryMT = ['audio/mp4', 'audio/mpeg', 'audio/ogg', 'audio/opus', 'audio/wav', 'audio/x-ms-wma', 'audio/webm'];

		checker = new Audio("");
		for(var i = 0; i <= 6; i++) {
			try {
				if(checker.canPlayType(queryMT[i]) != "") {
					if(audioCaps != "") audioCaps = audioCaps + "-";
					audioCaps = audioCaps + queryMT[i];
				}
			} catch (e) {
				// NOP
			}
		}
		return audioCaps;
	},

	/** prepare audio-object for re-use */
	resetAudio: function() {
		rau.pause();
		rau = new Audio();
	},

	/** tune station with multiple streams */
	tuneStation: function(id, streamData) {
		var source = document.createElement('source');

		$.each(streamData, function(key, values) {
			source.type = values.mimeType;
			source.src = values.streamUrl;
			rau.appendChild(source);
		});
		rau.load();
		if(rau) {
			rau.play();
			rau.volume = 1;
			stationName = rauData.queryStationData(id, "name");
			stationLongName = rauData.queryStationData(id, "longName");
			rauGui.setStation(rauData.selectStationName(stationName, stationLongName));
			rauGui.showStatus("");
		}
	},

	/** tune ONE stream */
	tuneStream: function(id, strid) {
		var source = document.createElement('source');
		var streamLabel = "";

		$.each(rauStreamData, function(key, values) {
			if(values.id == strid) {
				source.type = values.mimeType;
				source.src = values.streamUrl;
				rau.appendChild(source);
				
				streamLabel = rauGui.qualityTag(values.qualityBR);
				streamLabel = streamLabel + " " + rauGui.audioLabel(values.mimeType);
			}
		});
		rau.load();
		if(rau) {
			rau.play();
			rau.volume = 1;
			stationName = rauData.queryStationData(id, "name");
			stationLongName = rauData.queryStationData(id, "longName");
			rauGui.setStation(rauData.selectStationName(stationName, stationLongName));
			rauGui.setStream(streamLabel);
			rauGui.showStatus("");
		}
	},
}

/** radio GUI: controller */
var rauGuiCtrl = {

	/** startup the radio */
	startRadio: function() {
		rauData.loadStationData();
	},

	/** reset radio controls */
	resetGUI: function() {
		$("#streamButtons").html("<caption>Select a station...</caption>");
		$("#play").addClass("disabled");
		$("#mute").addClass("disabled");
	},

	/** show editor-link */
	showEditorLink: function() {
		rauData.loadEditorLink();
	},

	/** setup status event */
	setupStatus: function() {
		rau.addEventListener("timeupdate", function() {
			rauGui.showStatus(rau.currentTime);
			}, false
		);
	},

	/** setup presets: station-buttons */
	setupStations: function() {
		$(".rau-preset").each(function(x) {
			$(this).click(function() {
				rauDio.resetAudio();
				rauGui.setStation("");
				rauGui.setStream("");
				if($(this).attr("id") != "0") {rauShowOnceMSG = "reading database...";}
				rauGui.showStatus("");
				rauGuiCtrl.setupStatus();
				if($(this).attr("id") != "0") {rauData.loadStreamData($(this).attr("id"));}
			});
		});
	},
	
	/** setup presets: stream-buttons */
	setupStreams: function() {
		$(".rau-stream").each(function(x) {
			$(this).click(function() {
				rauDio.resetAudio();
				rauGui.setStation("");
				rauGui.setStream("");
				rauGui.showStatus("");
				rauGuiCtrl.setupStatus();
				rauDio.tuneStream($(this).attr("staid"), $(this).attr("strid"));
			});
		});
	},
	
	/** setup mute function */
	setupMute: function() {
		$("#play").click(function() {
			rau.volume = 1;
			$("#play").addClass("active");
			$("#mute").removeClass("active");
		});
		$("#play").addClass("active");
		$("#mute").click(function() {
			rau.volume = 0;
			$("#play").removeClass("active");
			$("#mute").addClass("active");
		});
	},
};

/** radio GUI: view */
var rauGui = {
	/** software info string */
	infoString: "v1 - 2020 Andreas Bahr",
	
	/** draw all station radio buttons */
	drawStationButtons: function() {
		var buttonHtml = "";
		
		buttonHtml = buttonHtml + "<thead><tr><th width=\"70\"><h4>Stations</h4></th><th>&nbsp;</th></tr></thead>";
		buttonHtml = buttonHtml + "<tbody><tr><td><center><button class=\"btn-large rau-preset\" id=\"0\" title=\"switch off\" type=\"button\" style=\"margin-top: 3px\"><span class=\"glyphicon glyphicon-off\" style=\"font-size: 300%;\"></span></button></center></td><td><h4><b>Offline</b></h4></td></tr>";
		$.each(rauStationData, function(key, values) {
			buttonHtml = buttonHtml + rauGui.drawStationTemplate(values.id, values.name, values.longName, values.nation, values.website, values.logoFile);
		});
		buttonHtml = buttonHtml + "</tbody>";
		return buttonHtml;
	},

	/** a stations button template */
	drawStationTemplate: function(stationId, stationName, stationLongName, stationNation, stationWebsite, stationLogoFile) {
		var html = ""; var fullName = "(Data missing)";
		
		fullName = rauData.selectStationName(stationName, stationLongName);
		if(stationLogoFile == null) {
			html = html + "<tr><td><center><button id=\"" + stationId + "\" title=\"click to play\" class=\"btn-large rau-preset\" type=\"button\" style=\"margin-top: 3px\"><span class=\"glyphicon glyphicon-play\" style=\"font-size: 300%;\"></span></button></center></td>";
		} else {
			html = html + "<tr><td><button id=\"" + stationId + "\" class=\"btn-large rau-preset\" type=\"button\"><img src=\"stationLogos/" + stationLogoFile + "\" alt=\"click to play\" width=\"64\" /></button></td>";
		}
		html = html + "<td><h4><b>" + fullName + "</b></h4>";
		if(stationNation != null) html = html + stationNation;
		if(stationNation != null && stationWebsite != null) html = html + " &bull; ";
		if(stationWebsite != null) html = html + "<a target=\"_blank\" href=\"http://" + stationWebsite + "\">" + stationWebsite + "</a>";
		html = html + "</td></tr>";
		return html;
	},

	/** draw all stream buttons */
	drawStreamButtons: function() {
		var buttonHtml = "";

		buttonHtml = buttonHtml + "<thead><tr><th width=\"70\"><h4>Streams</h4></th><th>&nbsp;</th></tr></thead>";
		$.each(rauStreamData, function(key, values) {
			buttonHtml = buttonHtml + rauGui.drawStreamTemplate(values.stationId, values.id, values.mimeType, values.qualityBR, values.streamUrl);
		});
		buttonHtml = buttonHtml + "</tbody>";
		return buttonHtml;
	},

	/** a streams button template */
	drawStreamTemplate: function(stationId, id, mimeType, qualityBR, streamUrl) {
		var html = "";

		html = html + "<tr><td><center><button staid=\"" + stationId + "\" strid=\"" + id + "\" title=\"click to play\" class=\"btn-large rau-stream\" type=\"button\" style=\"margin-top: 3px\"><span class=\"glyphicon glyphicon-play\" style=\"font-size: 300%;\"></span></button></center></td>";
		html = html + "<td style=\"line-height: 75%;\"><h4><b>" + rauGui.audioLabel(mimeType) + "</b>";
		html = html + ", " + rauGui.qualityLabel(qualityBR) + "</h4>";
		html = html + streamUrl + "</td></tr>";
		return html;
	},

	/** set the radio station label */
	setStation: function(station) {
		rauStation = station;
	},

	/** set the radio stream info */
	setStream: function(stream) {
		rauStream = stream;
	},

	/** show the radio status */
	showStatus: function(status) {
		if(rauStation != "") {
			if(status > 0) {
				$("#displayMain").text(rauStation);
				if(rauStream != "") {
					$("#displaySub").text(rauStream + ": streaming");
				} else {
					$("#displaySub").text("streaming");
				}
				$("#play").removeClass("disabled");
				$("#mute").removeClass("disabled");
			} else {
				$("#displayMain").text(rauStation);
				if(rauStream != "") {
					$("#displaySub").text(rauStream + ": loading...");
				} else {
					$("#displaySub").text("loading...");
				}
				$("#play").addClass("disabled");
				$("#mute").addClass("disabled");
			}
		} else {
			if(rauShowOnceMSG != "") {
				$("#displayMain").text("Offline");
				$("#displaySub").html(rauShowOnceMSG);
				$("#play").addClass("disabled");
				$("#mute").addClass("disabled");
				rauShowOnceMSG = "";
			} else {
				$("#displayMain").text("Offline");
				$("#displaySub").html("&nbsp;");
				$("#play").addClass("disabled");
				$("#mute").addClass("disabled");
			}
		}
	},
	
	/** show a link to the station-db-editor */
	showEditorLink: function(editorLink) {
		if(editorLink) {
			$("#editorLink").html("<a target=\"pentode-editor\" href=\"" + editorLink + "\"><span class=\"lead\">Station-Database-Editor</span></a>");	
		}
	},

	/** cleanup stream's list */
	cleanStreams: function() {
		$("#streamButtons").html("<caption>Select a station...</caption>");
		//$("#streamButtons").html("Select a station...");
	},

	/** mimetype text */
	audioLabel: function(mimeType) {
		var temp = "";
		temp = mimeType.substr(6);
		if(temp == "mpeg") {temp = "mp3";}
		if(temp == "x-ms-wma") {temp = "wma";}
		return temp.toUpperCase();
	},

	/** quality description text */
	qualityLabel: function(qualityBR) {
		switch(parseInt(qualityBR)) {
			case 1:
				return "Best Quality, high Bitrate";
				break;
			case 2:
				return "Medium Quality / Bitrate";
				break;
			case 3:
				return "Low Quality / Bitrate";
				break;
			default:
				return "Unknown Quality";
		}
	},

	/** quality description tag */
	qualityTag: function(qualityBR) {
		switch(parseInt(qualityBR)) {
			case 1:
				return "Best";
				break;
			case 2:
				return "Medium";
				break;
			case 3:
				return "Low";
				break;
			default:
				return "Unknown";
		}
	},
};

/** raudio toolbox */
var rauToolbox = {

	/** client/browser name - thanxs alot mozilla */
	clientName: function() {
		var sBrowser, sUsrAg = navigator.userAgent;

		if (sUsrAg.indexOf("Firefox") > -1) {
		  sBrowser = "Firefox";
		} else if (sUsrAg.indexOf("Opera") > -1 || sUsrAg.indexOf("OPR") > -1) {
		  sBrowser = "Opera";
		} else if (sUsrAg.indexOf("Trident") > -1) {
		  sBrowser = "IEX";
		} else if (sUsrAg.indexOf("Edge") > -1) {
		  sBrowser = "Edge";
		} else if (sUsrAg.indexOf("Chrome") > -1) {
		  sBrowser = "Chrome";
		} else if (sUsrAg.indexOf("Safari") > -1) {
		  sBrowser = "Safari";
		} else {
		  sBrowser = "Unknown";
		}
		return sBrowser;
	},

	/** client OS - thanxs alot mozilla */
	clientOS: function() {
		var sOS, sUsrAg = navigator.userAgent;

		if (sUsrAg.indexOf("Linux") > -1) {
		  sOS = "Linux";
		} else if (sUsrAg.indexOf("Windows") > -1) {
		  sOS = "Windows";
		} else if (sUsrAg.indexOf("Trident") > -1) {
		  sOS = "Macintosh";
		} else if (sUsrAg.indexOf("Edge") > -1) {
		  sOS = "Macintosh";
		} else if (sUsrAg.indexOf("iPhone") > -1) {
		  sOS = "iPhone";
		} else if (sUsrAg.indexOf("Android") > -1) {
		  sOS = "Android";
		} else {
		  sOS = "Unknown";
		}
		return sOS;
	},
};

